#ifndef LLF_ECPKI_BUILD_H
#define LLF_ECPKI_BUILD_H

/*
* All the includes that are needed for code using this module to
* Compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_ECPKI_BUILD.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
* Creation date : Tues Mar 03 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains declaration of low level function
* for export/import ECC key functions, which worked with 
* LibTomCrypt. 
*
* \version LLF_ECPKI_BUILD.h#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_ECPKI_BuildPrivKey
*
* Inputs:
* @param DomainID [in] - The enumerator variable defines current EC domain.
* @param PrivKeyIn_ptr [in] - Pointer to private key data.
* @param PrivKeySizeInBytes [in] - Size of private key data in bytes.
* @param UserPrivKey_ptr [out] - Pointer to the private key structure. 
*                    This structure is used as input to the ECPKI 
*                    cryptographic primitives.
*                    Size of buffer must be not less than:
*                    Order size of the appropriative domain in bytes
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  Builds (imports) the user private key structure 
*  from an existing private key so that this structure 
*  can be used by other EC primitives.
*
* \b 
* Algorithm:
*  -# Initialize LibTomCrypt primitives;
*  -# Read ECC private scalar value;
*  -# Make the public key;
*  -# Export LibTomCrypt ECC key to CE2 key format.
***************************************************************/
CE2Error_t LLF_ECPKI_BuildPrivKey(
    CE2_ECPKI_DomainID_t      DomainID,	      /*in */   
    DxUint8_t			      *PrivKeyIn_ptr,     /*in*/
    DxUint32_t                 PrivKeySizeInBytes,/*in*/
    CE2_ECPKI_UserPrivKey_t  *UserPrivKey_ptr    /*out*/ );

/**
****************************************************************
* Function Name: 
*  LLF_ECPKI_BuildPublKey
*
* Inputs:
* @param DomainID [in] - The enumerator variable defines current EC domain.
* @param PublKeyIn_ptr [in] - Pointer to public key data.
* @param PublKeySizeInBytes [in] - Size of public key data in bytes.
* @param UserPublKey_ptr [out] - Pointer to the public key structure. 
*                    This structure is used as input to the ECPKI 
*                    cryptographic primitives.
*                    Size of buffer must be not less than:
*					 2*ModSizeInBytes - for WM DRM,
*					 2*ModSizeInBytes+1 - for other domains.  
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  Builds (imports) the user public key structure 
*  from an existing public key so that this structure 
*  can be used by other EC primitives.
*
* \b 
* Algorithm:
*  -# Determinate key size, initialize key structures;
*  -# Read ECC public key point;
*  -# Set other LibTomCrypt ECC key parameters;
*  -# Export LibTomCrypt ECC key to CE2 key format.
***************************************************************/
CE2Error_t LLF_ECPKI_BuildPublKey(
    CE2_ECPKI_DomainID_t      DomainID,	        /*in*/				
    DxUint8_t			      *PublKeyIn_ptr,       /*in*/
    DxUint32_t                 PublKeySizeInBytes,  /*in*/
    CE2_ECPKI_UserPublKey_t  *UserPublKey_ptr      /*out*/ );

/**
****************************************************************
* Function Name: 
*  LLF_ECPKI_ExportPublKey
*
* Inputs:
* @param UserPublKey_ptr [in] - A pointer to the public key structure 
*                    (in little-endian form). 
* @param Compression [in] - An enumerator parameter: defines point compression.
* @param ExternPublKey_ptr [out] - A pointer to the exported public key structure 
*                      in big-endian and point compression as defined 
*                      by input parameter. 
*                      Size of buffer must be not less than:
*					   2*ModSizeInBytes - for WM DRM,
*					   2*ModSizeInBytes+1 - for other domains.   
* @param PublKeySizeInBytes [in/out] - A pointer to a variable for input size of 
*                       user passed buffer for public key and output 
*                       the size of converted public key in bytes.
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  Converts an existing public key in little-endian form to  
*  big-endian export form and initializes an appropriate structure 
*  of type CE2_ECPKI_PublKeyExtern_t.
*
* \b 
* Algorithm:
*  -# Import CE2 user public key to LibTomCrypt key structure;
*  -# Write ECC public key point to temporary buffer 
*    (in big-endian format);
*  -# Make export public key (compress Y coordinate if need).
***************************************************************/
CE2Error_t LLF_ECPKI_ExportPublKey(
    CE2_ECPKI_UserPublKey_t *UserPublKey_ptr,     /*in*/
    CE2_ECPKI_PointCompression_t  Compression,         /*in*/
    DxUint8_t	*ExternPublKey_ptr,   /*in*/
    DxUint32_t *PublKeySizeInBytes   /*in/out*/ );

/**
****************************************************************
* Function Name: 
*  LLF_ECPKI_ExportPrivKey
*
* Inputs:
* @param UserPrivKey_ptr [in] - A pointer to the private key structure 
*                    (in little-endian form). 
* @param ExternPrivKey_ptr [out] - A pointer to the exported private key 
*                      in big-endian. 
*						Size of buffer must be not less than:
*						Order size of the appropriative domain in bytes
* @param ExternPrivKeySize_ptr [in\out]- A pointer to a variable for 
*                       input size of user passed buffer for private key 
*                       and output the size of converted private key in bytes.
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_USER_PRIV_KEY_PTR_ERROR
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_PTR_ERROR
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  Converts an existing private key in little-endian form to  
*  big-endian export form.
*
* \b 
* Algorithm:
*  -# Import CE2 user private key to LibTomCrypt key structure;
*  -# Write ECC private scalar value to output buffer 
*    (in big-endian format);
***************************************************************/
CE2Error_t LLF_ECPKI_ExportPrivKey(
  CE2_ECPKI_UserPrivKey_t       *UserPrivKey_ptr,     /*in*/
  DxUint8_t			                *ExternPrivKey_ptr,   /*out*/
  DxUint32_t                    *ExternPrivKeySize_ptr   /*in/out*/ );

#ifdef __cplusplus
}
#endif

#endif
